/*
 * Copyright (c) 2004 John Topley (www.johntopley.com)
 * Project      : Web Forum
 * File         : UsersDAO.java
 * Classes      : UsersDAO 
 */
package com.johntopley.webforum.data;

import com.johntopley.webforum.data.BaseDAO;
import com.johntopley.webforum.model.User;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.naming.NamingException;

/**
 * Data Access Object for <code>User</code> and <code>Users</code> objects.
 * 
 * @author John Topley
 * @version 1.0
 * 
 * @see com.johntopley.webforum.model.User
 * @see com.johntopley.webforum.model.Users
 */
public class UsersDAO extends BaseDAO
{
  /**
   * No argument constructor for creating a new <code>UsersDAO</code>. Calls the
   * superclass constructor.
   * 
   * @throws NamingException if thrown by the superclass.
   * 
   * @see com.johntopley.webforum.data.BaseDAO#BaseDAO()
   */
  public UsersDAO() throws NamingException
  {
    super();
  }

  /**
   * Gets the number of posts made by the specified user.
   * 
   * @param userID The unique ID of the user to get the count of posts for.
   * @return a <code>Long</code> containing the number of posts made by the
   * specified user.
   * @throws SQLException if the post count could not be retrieved.
   */
  public Long getPostCount(Long userID) throws SQLException
  {
    if (userID == null || userID.longValue() == 0)
    {
      throw new IllegalArgumentException("userID cannot be null or 0.");
    }

    Connection conn = null;
    PreparedStatement pstmt = null;
    ResultSet rset = null;
    Long postCount = new Long(0);

    StringBuffer sql = new StringBuffer();
    sql.append("SELECT  COUNT(p.PostID) AS PostCount ")
       .append("FROM    Posts p ")
       .append("WHERE   p.UserID = ?");
    
    try
    {
      conn = super.getConnection();
      pstmt = conn.prepareStatement(sql.toString());
      pstmt.setLong(1, userID.longValue());
      rset = pstmt.executeQuery();

      while (rset.next())
      {
        postCount = new Long(rset.getInt("PostCount"));
      }      
    }    
    finally
    {
      super.closeResources(rset, pstmt, conn);
    }

    return postCount;
  }

  /**
   * Gets a <code>User</code> object for the specified user ID.
   * 
   * @param userID The unique ID of the user.
   * @return a <code>User</code> object representing the user with the specified
   * ID.
   * @throws SQLException if the user's details could not be retrieved.
   */
  public User getUser(Long userID) throws SQLException
  {
    if (userID == null || userID.longValue() == 0)
    {
      throw new IllegalArgumentException("userID cannot be null or 0.");
    }

    Connection conn = null;
    PreparedStatement pstmt = null;
    ResultSet rset = null;
    User user = null;

    StringBuffer sql = new StringBuffer();
    sql.append("SELECT   u.Username ")
       .append("       , u.Forenames ")
       .append("       , u.Surname ")
       .append("FROM     Users u ")
       .append("WHERE    u.UserID = ?");       
    
    try
    {
      conn = super.getConnection();
      pstmt = conn.prepareStatement(sql.toString());
      pstmt.setLong(1, userID.longValue());
      rset = pstmt.executeQuery();

      while (rset.next())
      {
        String username = rset.getString("u.Username");
        String forenames = rset.getString("u.Forenames");
        String surname = rset.getString("u.Surname");
        Long postCount = this.getPostCount(userID);

        user = new User(userID.toString(),
                        username,
                        forenames,
                        surname,
                        postCount.toString());        
      }
    }    
    finally
    {
      super.closeResources(rset, pstmt, conn);
    }

    return user;
  }  
}